unit SVis;

{ === SVis unit ===}
{Author(s): J. Frankel,  N.M. Ismail (11/16/97),               }
{           J. Crawford (1/25/98)                              }
{Version: See =History= below.                                 }
{Description: This unit is the sample visualization plug-in    }
{   for a WinAMP using the basic structures that are featured  }
{   in the Vis unit.  Standard WINAPI calls are used instead of}
{   Delphi's "TForm"s.  (Possible next release to have TForms?)}
{Usage: Study the methods used below and create a plug-in based}
{   off of the Vis unit's structures.                          }

{ === History === }
{  Date    |  Author      | Version | Notes                    }
{  1997    | J. Frankel   |  1.01   | 1st release in VC++ code.}
{ 11/16/97 | N.M. Ismail  |  1.01d  | Converted from original  }
{          |              |         | VC++ source to Delphi2/3.}
{ 1/26/98  | J. Crawford  |  1.1    | Additional testing and   }
{          |              |         | bugfixes.  Better docs.  }
{          |              |         | Bundled original C code. }

{ === Notes === }
{ This source is freeware, so fiddle with it all you want;     }
{ Just keep it freeware.  (Or else.)                           }

interface

uses
    MySysUtl, Windows, Vis;

{ === Exported declarations === }
function winampVisGetHeader : PwinampVisHeader; cdecl; export;
         {this is the only exported symbol.}
         {returns our main header.}

{ === Configuration declarations === }
const
     szAppName :PChar= 'SimpleVis'; {Our window class}
     config_x  :integer= 50;
     config_y  :integer= 50;
      {screen X position and Y position, respectively}

var
  hMainWnd  :HWND; {main window handle}
{ Double buffering data }
  memDC  :HDC;     {memory device context}
  memBM,           {memory bitmap (for memDC)}
  oldBM  :HBITMAP; {old bitmap (from memDC)}

{ === Forward declarations === }
function  GetModule(Which :integer) :PWinAMPVisModule; cdecl;
          {returns a PWinAMPVisModule when requested.}
          {Used in the HDR (header) below}
procedure Config(This_Mod : PWinAMPVisModule); cdecl;
          {configuration dialog}
function  Init(This_Mod : PWinAMPVisModule) :integer; cdecl;
          {generic initialization for module}
function  Render1(This_Mod : PWinAMPVisModule) :integer; cdecl;
          {specialized rendering for module 1}
function  Render2(This_Mod : PWinAMPVisModule) :integer; cdecl;
          {specialized rendering for module 2}
function  Render3(This_Mod : PWinAMPVisModule) :integer; cdecl;
          {specialized rendering for module 3}
procedure Quit(This_Mod : PWinAMPVisModule); cdecl;
          {generic de-initialization for module}

{ === Plug-in headers and modules === }
const
{ === Module header: includes version, description, === }
{ === and address of the module retriever function  === }
     HDR  :TWinAMPVisHeader =
           (Version      :VIS_HDRVER;
            Description  :'Nullsoft Test Visualization Library v1.10';
            GetModule    :GetModule);
{ === Modules: description and methods === }
     Mod1 :TWinAMPVisModule = {1st module (oscilliscope)}
           (Description  :'Oscilliscope';
            hWNDParent   :0; {filled in by WinAMP}
            hDLLInstance :0; {filled in by WinAMP}
            sRate        :0; {filled in by WinAMP}
            nCh          :0; {filled in by WinAMP}
            LatencyMs    :25;
            DelayMS      :25;
            SpectrumNch  :0;
            WaveformNch  :2;
            Config       :Config;
            Init         :Init;
            Render       :Render1;
            Quit         :Quit;
            UserData     :nil);
     Mod2 :TWinAMPVisModule = {2nd module (spectrum analyzer)}
           (Description  :'Spectrum Analyzer';
            hWNDParent   :0; {filled in by WinAMP}
            hDLLInstance :0; {filled in by WinAMP}
            sRate        :0; {filled in by WinAMP}
            nCh          :0; {filled in by WinAMP}
            LatencyMs    :25;
            DelayMS      :25;
            SpectrumNch  :2;
            WaveformNch  :0;
            Config       :Config;
            Init         :Init;
            Render       :Render2;
            Quit         :Quit;
            UserData     :nil);
     Mod3 :TWinAMPVisModule = {3rd module (VU meter)}
           (Description  :'VU Meter';
            hWNDParent   :0; {filled in by WinAMP}
            hDLLInstance :0; {filled in by WinAMP}
            sRate        :0; {filled in by WinAMP}
            nCh          :0; {filled in by WinAMP}
            LatencyMs    :25;
            DelayMS      :25;
            SpectrumNch  :0;
            WaveformNch  :2;
            Config       :Config;
            Init         :Init;
            Render       :Render3;
            Quit         :Quit;
            UserData     :nil);

implementation

{ === Forward declarations === }
procedure ConfigRead(This_Mod :PWinAMPVisModule); forward;
          {reads the configuration}
procedure ConfigWrite(This_Mod :PWinAMPVisModule); forward;
          {writes the configuration}
procedure ConfigGetINI_FN(This_Mod :PWinAMPVisModule; INIFile :PChar); forward;
          {makes the .ini file filename}

{ === Our routines for ease of use ===}

function  OurMessageBox(Owner :HWND; Msg, Title :string; Style :integer) :integer;
          {Makes a message box with a style of STYLE}
          {MB_OK = OK button}
          {MB_OKCANCEL = OK and Cancel buttons}
          {Returns an ID (0 = ID_OK)}
          begin
          Msg := Msg + #0;
          Title := Title + #0;
          Result := MessageBox(Owner,@Msg[1],@Title[1],Style);
          end;

{ === Visualization plug-in methods === }

function  WinAMPVisGetHeader :PWinAMPVisHeader;
          {WinAMP calls this exported function}
          {to get the plug-in header}
begin
  Result := @HDR;  {Return the main header}
end; { winampVisGetHeader }

function  GetModule(Which :integer) :PwinampVisModule;
          {GetModule routine from the main header.}
          {Returns nil if an invalid module was requested;}
          {otherwise returns the corresponding module pointer}
          {depending on which.}
begin
  case which of
    0 : Result := @Mod1;
    1 : Result := @Mod2;
    2 : Result := @Mod3;
   else Result := nil;
  end;
end; { GetModule }

procedure Config(This_Mod :PWinAMPVisModule);
          {The configuration method for the given module}
          {Doesn't have to be a single procedure that shares}
          {all the modules.  You can have multiple Configs:}
          {Config1, Config2, Config3, etc.}
begin
  OurMessageBox(This_Mod^.hWNDParent,
                'This module is Copyright(C) 1997, Justin Frankel/Nullsoft'#13+
                'Delphi conversion by N.M. ISMAIL - 11/16/97'#13+
                'Restructuring for Delphi by Jason Crawford - 1/26/98'#13+
                '-- This is just a demonstration module, it really isn''t'#13+
                '   supposed to be enjoyable --',
                'Configuration: '+This_Mod^.Description,
                MB_OK);
end; { Config }

function  Init(This_mod :PWinAMPVisModule) :integer;
          {Registers our window class, creates our window, etc.}
          {Again, this one works for all modules, but you could}
          {make init1, init2, init3, etc...}
          {Returns 0 on success; 1 on failure.}
          var
             width, height :integer;
             wc            :TWndClass;
             r             :TRect;
             wcHandle      :integer;
             _hdc          :HDC;
begin
  {width and height are the same for Mod1 and Mod2,}
  {but mod3 is shaped differently}
  if This_Mod = @mod3 then
  begin
    width := 256;
    height := 32;
  end else
  begin
    width := 288;
    height := 256;
  end;

  ConfigRead(This_Mod);  {Read the configuration for the Module}

  {Register our window class}
  with wc do
  begin
    wc.style := 0;   {no special style for this class}
    wc.lpfnWndProc := @DefWindowProc;  {Our window procedure}
    wc.cbClsExtra := 0;  {Cleared extra parameters}
    wc.cbWndExtra := 0;  {Extra window data in bytes}
    wc.hInstance := This_Mod^.hDllInstance;  {hInstance of DLL}
    wc.hIcon := 0;  {The special Icon of the window; 0=none}
    wc.hCursor := 0;  {The special Cursor of the window; 0=none}
    wc.hbrBackground := 0; {The background style}
    wc.lpszMenuName := #0; {The menu name (none=#0)}
    wc.lpszClassName := szAppName; {Our window class name}
  end;
  wcHandle := Windows.RegisterClass(wc);
  if wcHandle = 0 then
  begin
    OurMessageBox(This_Mod^.hWNDParent,
                  'Error registering window class',
                  This_Mod^.Description,
                  MB_OK);
    Result:=1;
    exit;
  end;

  {Create the window}
  hMainWnd := CreateWindowEx(
              WS_EX_TOOLWINDOW or WS_EX_APPWINDOW,
                {these exstyles put a nice small frame,}
                {but also a button in the taskbar}
              szAppName,
                {our window class name}
              This_Mod^.Description,
                {use description for a window title}
              WS_VISIBLE,
                {make the window visible}
                {Note: we used to have a}
                {close button, but we don't}
                {have the closed window}
                {detection routine in yet.}
              config_x, config_y,
                {screen position (read from config)}
              width, height,
                {width & height of window}
                {(need to adjust client area later)}
              This_Mod^.hWNDParent,
                {parent window (WinAMP main window)}
              0,
                {no menu}
              This_Mod^.hDLLInstance,
                {hInstance of DLL}
              nil);
                {no window creation data}
  if (hMainWnd = 0) then
  begin
    OurMessageBox(This_Mod^.hWNDParent,
                  'Error creating window',
                  this_mod^.description,
                  MB_OK);
    Result:=1;
    exit;
  end;

  {set our window "user data" to the "This_Mod" pointer}
  SetWindowLong(hMainWnd,GWL_USERDATA,longint(This_Mod));

  {adjust size of window to make the client area
  {exactly width x height}
  GetClientRect(hMainWnd,r);
  SetWindowPos(hMainWnd,0,0,0,width*2-r.right,
               height*2-r.bottom,SWP_NOMOVE or SWP_NOZORDER);

  {create our doublebuffer}
  memDC := CreateCompatibleDC(0);
  memBM := CreateCompatibleBitmap(memDC,width,height);
  oldBM := SelectObject(memDC,memBM);

  {show the window}
  ShowWindow(hMainWnd,SW_SHOWNORMAL);

  {clear background}
  Rectangle(memDC,0,0,width,height);

  {copy doublebuffer to window}
  _hdc := GetDC(hMainWnd);
  BitBlt(_hdc,0,0,width,height,memDC,0,0,SRCCOPY);
  ReleaseDC(hMainWnd,_hdc);

  Result := 0;
end; { Init }

function  Render1(This_Mod :PWinAMPVisModule) :integer;
          {render function for oscilliscope.}
          {Returns 0 if successful, 1 if visualization}
          {should end.}
          var
             x, y :integer;
             _hdc :HDC;
begin
  {clear background}
  Rectangle(memDC,0,0,288,256);

  {draw oscilliscope}
  for y := 0 to This_Mod^.nCh -1 do
  begin
    MoveToEx(memDC,0,(y*256) shr (this_mod^.nCh-1),nil);
    for x := 0 to 288-1 do
      LineTo(memDC,x,(y*256 + (ord(this_mod^.waveformData[y][x]) xor 128)) shr (this_mod^.nCh-1));
  end;

  {copy doublebuffer to window}
  _hdc := GetDC(hMainWnd);
  BitBlt(_hdc,0,0,288,256,memDC,0,0,SRCCOPY);
  ReleaseDC(hMainWnd,_hdc);

  Result := 0;
end; { Render1 }

function  Render2(This_Mod : PWinAMPVisModule) :integer;
          {render function for analyser.}
          {Returns 0 if successful, 1 if visualization}
          {should end.}
          var
             x, y :integer;
             _hdc :HDC;
begin
  {clear background}
  Rectangle(memDC,0,0,288,256);

  {draw analyser}
  for y := 0 to this_mod^.nCh-1 do
    for x := 0 to 288-1 do
    begin
      MoveToEx(memDC,x,(y*256+256) shr (this_mod^.nCh-1),nil);
      LineTo(memDC,x,(y*256 + 256 - ord(this_mod^.spectrumData[y][x])) shr (this_mod^.nCh-1));
    end;

  {copy doublebuffer to window}
  _hdc := GetDC(hMainWnd);
  BitBlt(_hdc,0,0,288,256,memDC,0,0,SRCCOPY);
  ReleaseDC(hMainWnd,_hdc);

  Result := 0;
end; { Render2 }

function  Render3(This_Mod : PWinAMPVisModule) :integer;
          {render function for VU meter.}
          {Returns 0 if successful, 1 if visualization}
          { should end.}
          var
             x, y        :integer;
             last, total :integer;
             _hdc        :HDC;
begin
  {clear background}
  Rectangle(memDC,0,0,256,32);

  {draw VU meter}
  for y := 0 to 2-1 do
  begin
    last := ord(this_mod^.waveformData[y][0]);
    total := 0;
    for x := 1 to 576-1 do
    begin
      inc(total, abs(last - ord(this_mod^.waveformData[y][x])));
      last:=ord(this_mod^.waveformData[y][x]);
    end;
    total := total div 288;
    if total > 127 then
      total := 127;
    if y <> 0 then
      Rectangle(memDC,128,0,128+total,32) else
      Rectangle(memDC,128-total,0,128,32);
  end;

  {copy doublebuffer to window}
  _hdc := GetDC(hMainWnd);
  BitBlt(_hdc,0,0,256,32,memDC,0,0,SRCCOPY);
  ReleaseDC(hMainWnd,_hdc);

  Result := 0;
end; { Render3 }

procedure Quit(This_Mod :PWinAMPVisModule);
          {Cleanup (opposite of Init).}
          {Destroys the window, unregisters the window class}
begin
  ConfigWrite(This_Mod);     {write configuration}
  SelectObject(memDC,oldBM); {delete our doublebuffer}
  DeleteObject(memDC);
  DeleteObject(memBM);
  DestroyWindow(hMainWnd);    {delete our window}
  Windows.UnregisterClass(szAppName,This_Mod^.hDLLInstance);
    {unregister window class}
end; { Quit }

procedure ConfigGetINI_FN(This_Mod :PWinAMPVisModule; INIFile :PChar);
          {makes an .INI file in the winamp directory}
          {named "plugin.ini"}
          var
             p :PChar;
begin
  GetModuleFileName(This_Mod^.hDLLInstance,INIFile,MAX_PATH);
  p := INIFile + StrLen(INIFile);
  while (p >= INIFile) and (p^ <> '\') do
    dec(p);
  inc(p);
  if p >= INIFile then
    p^ := #0;
  StrCat(INIFile,'plugin.ini');
end; { ConfigGetINI_FN }


procedure ConfigRead(This_Mod :PWinAMPVisModule);
          var
             INIFile : array[0..MAX_PATH-1] of char;
begin
  ConfigGetINI_FN(This_Mod,INIFile);
  config_x := GetPrivateProfileInt(This_Mod^.Description,
                                   'Screen_x',
                                   config_x,
                                   INIFile);
  config_y := GetPrivateProfileInt(This_Mod^.Description,
                                   'Screen_y',
                                   config_y,
                                   INIFile);
end; { ConfigRead }

procedure ConfigWrite(This_Mod :PWinAMPVisModule);
          var
             _string :array[0..31] of char;
             INIFile :array[0..MAX_PATH-1] of char;
begin
  ConfigGetINI_FN(This_Mod,INIFile);
  str(config_x, _string);  {More efficient than StrFmt(...)}
  WritePrivateProfileString(This_Mod^.Description,
                            'Screen_x',
                            _string,
                            INIFile);
  str(config_y, _string);
  WritePrivateProfileString(This_Mod^.Description,
                            'Screen_y',
                            _string,
                            INIFile);
end; { ConfigWrite }

end.
